<?php

namespace App\Http\Controllers;

use App\Http\Requests\Role\DeleteRoleRequest;
use App\Http\Requests\Role\StoreRoleRequest;
use App\Http\Requests\Role\UpdateRoleRequest;
use App\Http\Requests\Role\ViewRoleRequest;
use App\Models\Role;
use App\Models\User;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\ResponseTools;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\DB;
use Throwable;

class RoleController extends Controller
{
    public function index(ViewRoleRequest $request)
    {
        try {
            $dataTableTools = new DataTableTools();
            $query = Role::select();
            $queryFull = Role::with('permissions');

            $dataTableTools->doQuery([$query, $queryFull], $request)
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('roles', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $exception) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'نقش'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function store(StoreRoleRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $roleAdd = CommonTools::safeRequest($request, 'role');

            DB::beginTransaction();

            $role = Role::create(CommonTools::toSafeArray($roleAdd, 'name', 'title'));

            $role->permissions()->sync($roleAdd['permissions']);

            $user->appendLog($request, ['new' => $role]);
            DB::commit();

            $role->load(['permissions']);

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_recorded',
                    ['title' => 'نقش']
                )
                ->setData('role', $role)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'roleStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_recording_information',
                ['title' => 'نقش']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function destroy(DeleteRoleRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $role = Role::find($request->get('id'));

            DB::beginTransaction();

            $user->appendLog($request, ['old' => $role->toArray()]);
            $role->delete();

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_deleted',
                    ['title' => 'نقش']
                )
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'roleDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat(
                'messages.error_deleting_information',
                ['title' => 'نقش']
            )
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdateRoleRequest $request)
    {

        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->setData('errors', $request->errorsToArray())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $requestBody = CommonTools::safeRequest($request, 'id', 'role');
            $roleUpdate = $requestBody['role'];
            $roleOriginal = Role::find($requestBody['id']);

            DB::beginTransaction();

            $user->appendLog($request, [
                'old' => $roleOriginal->toArray(),
                'new' => $roleUpdate,
            ]);
            $roleOriginal->update(CommonTools::toSafeArray($roleUpdate, 'name', 'title'));

            $roleOriginal->permissions()->sync($roleUpdate['permissions']);

            DB::commit();

            $roleOriginal->load(['permissions']);

            return ResponseTools::getInstance()
                ->setMessageFormat(
                    'messages.information_was_successfully_updated',
                    ['title' => 'نقش']
                )
                ->setData('role', $roleOriginal)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'roleUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_updating_information', ['title' => 'نقش'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}
