<?php

namespace App\Http\Controllers;

use App\Enums\PermissionStudentAccess;
use App\Http\Requests\Student\DeleteStudentRequest;
use App\Http\Requests\Student\ExcelStoreStudentRequest;
use App\Http\Requests\Student\PrivateStorageStudentRequest;
use App\Http\Requests\Student\StoreStudentRequest;
use App\Http\Requests\Student\UpdateStudentRequest;
use App\Http\Requests\Student\ViewOwnStudentRequest;
use App\Http\Requests\Student\ViewStudentRequest;
use App\Imports\StudentImport;
use App\Models\Student;
use App\Models\User;
use App\Tools\CommonTools;
use App\Tools\DataTableTools;
use App\Tools\FileTools;
use App\Tools\ResponseTools;
use DB;
use Excel;
use Illuminate\Support\Arr;
use Maatwebsite\Excel\Validators\ValidationException;
use Symfony\Component\HttpFoundation\Response;
use Throwable;
use Storage;

class StudentController extends Controller
{

    public function index(ViewStudentRequest $request)
    {
        try {
            $user = User::find(auth()->user()->id ?? -1);
            $hasPermission = $user->hasPermissions([
                PermissionStudentAccess::view->value
            ]);

            $dataTableTools = new DataTableTools();

            $query = Student::select();

            if ($hasPermission) {
                $query->with([
                    'studentParent:id',
                    'studentParent.profile:id,user_id,first_name,last_name,address,phone',
                    'classRooms:id,academic_level_id,major_id,title,code',
                    'image',
                ]);
            } else {
                $teacherUserId = $user->id;
                $query->with([
                    'studentParent:id',
                    'studentParent.profile:id,user_id,first_name,last_name',
                    'classRooms:id,academic_level_id,major_id,title,code',
                    'image',
                ])
                    ->whereHas(
                        'classRooms',
                        function ($qClassRoom) use ($teacherUserId) {
                            $qClassRoom->whereHas(
                                'weeklySchedules',
                                function ($qWeeklySchedule) use ($teacherUserId) {
                                    $qWeeklySchedule->where(
                                        'teacher_id',
                                        '=',
                                        $teacherUserId
                                    );
                                }
                            );
                        }
                    );
            }

            $dataTableTools->doQuery($query, $request)
                ->appendFields(['current_class_room_id'])
                ->makeHidden(['classRooms.pivot'])
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('students', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $e) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'دانش آموز'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }


    public function me(ViewOwnStudentRequest $request)
    {
        try {
            $user = User::find(auth()->user()->id ?? -1);
            $dataTableTools = new DataTableTools();

            $query = $user->children()
                ->with([
                    'studentParent:id',
                    'studentParent.profile:id,user_id,first_name,last_name,address,phone',
                    'classRooms:id,academic_level_id,major_id,title,code',
                    'image',
                ]);

            $dataTableTools->doQuery($query, $request)
                ->appendFields(['current_class_room_id'])
                ->makeHidden(['classRooms.pivot'])
                ->toArray();

            if ($dataTableTools->status == ResponseTools::RES_SUCCESS) {
                return ResponseTools::getInstance()
                    ->setMessage(__('messages.information_was_successfully_get'))
                    ->setCount($dataTableTools->count)
                    ->setData('students', $dataTableTools->data)
                    ->getJsonResponse();
            } elseif (
                $dataTableTools->status == ResponseTools::RES_WARNING
            ) {

                return ResponseTools::getInstance()
                    ->setStatus(ResponseTools::RES_WARNING)
                    ->setMessage($dataTableTools->message)
                    ->getJsonResponse();
            }

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($dataTableTools->message)
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $e) {
            //ignore catch
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_get_information', ['title' => 'دانش آموز'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function store(StoreStudentRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        $file = null;
        try {
            $user = User::find(auth()->user()->id ?? -1);
            $studentAdd = CommonTools::safeRequest($request, 'student');
            $classRoomId = Arr::pull($studentAdd, 'class_room_id') ?? null;

            if (isset($studentAdd['image'])) {
                $file = (new FileTools())
                    ->privateDir()
                    ->uploadFile(
                        FileTools::$STORE_STUDENT,
                        $studentAdd['image']
                    )
                    ->store();
            }

            DB::beginTransaction();

            $student = Student::create($studentAdd);

            if ($file != null) {
                $student->image()
                    ->create([
                        'name' => $file->getFullName(),
                    ]);
            }

            if ($classRoomId != null) {
                $student->classRooms()
                    ->attach($classRoomId);
            }

            $user->appendLog($request, ['new' => $student]);
            DB::commit();

            $student->load([
                'studentParent:id',
                'studentParent.profile:id,user_id,first_name,last_name,address,phone',
                'classRooms:id,academic_level_id,major_id,title,code',
                'image',
            ]);

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_recorded', ['title' => 'دانش آموز'])
                ->setData('student',  $student)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            if ($file) {
                $file->removeFile();
            }

            CommonTools::registerException($e, 'studentStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_recording_information', ['title' => 'دانش آموز'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }


    public function excelStore(ExcelStoreStudentRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $studentsAdd = CommonTools::safeRequest($request, 'xlsx');

            DB::beginTransaction();

            Excel::import(
                new StudentImport(),
                $studentsAdd
            );

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_recorded', ['title' => 'دانش آموز'])
                ->getJsonResponse();
        } catch (ValidationException $e) {
            DB::rollBack();
            CommonTools::registerException($e, 'studentExcelStore');

            $messageErrors = $e->errors();

            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage(array_slice($messageErrors, 0, min(15, count($messageErrors))))
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        } catch (Throwable $e) {
            DB::rollBack();

            CommonTools::registerException($e, 'studentExcelStore');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_recording_information', ['title' => 'دانش آموز'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function destroy(DeleteStudentRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $student = Student::find($request->get('id'));

            DB::beginTransaction();

            $safeStudentImage = $student->image->name ?? null;

            $student->image()
                ->delete();

            $user->appendLog($request, ['old' => $student->toArray()]);
            $student->delete();

            if (isset($safeStudentImage)) {
                (new FileTools())
                    ->privateDir()
                    ->loadFromStudent($safeStudentImage)
                    ->removeFile();
            }

            DB::commit();

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_deleted', ['title' => 'دانش آموز'])
                ->getJsonResponse();
        } catch (Throwable $exception) {
            DB::rollBack();

            CommonTools::registerException($exception, 'studentDelete');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_deleting_information', ['title' => 'دانش آموز'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function update(UpdateStudentRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        $fileUpdate = null;

        try {
            $user = User::find(auth()->user()->id ?? -1);
            $requestBody = CommonTools::safeRequest($request, 'id', 'student');
            $studentUpdate = $requestBody['student'];
            $studentOriginal = Student::find($requestBody['id']);
            $classRoomId = Arr::pull($studentUpdate, 'class_room_id') ?? null;
            $deleteClassRoomIds = Arr::pull($studentUpdate, 'delete_class_room_ids') ?? null;

            if (isset($studentUpdate['image'])) {
                $fileUpdate = (new FileTools())
                    ->privateDir()
                    ->uploadFile(
                        FileTools::$STORE_STUDENT,
                        $studentUpdate['image']
                    )
                    ->store();
            }

            DB::beginTransaction();

            if ($fileUpdate) {
                $safeStudentImage = $studentOriginal->image->name ?? null;

                $studentOriginal->image()
                    ->delete();

                $studentOriginal->image()
                    ->create([
                        'name' => $fileUpdate->getFullName(),
                    ]);

                if (isset($safeStudentImage)) {
                    (new FileTools())
                        ->privateDir()
                        ->loadFromStudent($safeStudentImage)
                        ->removeFile();
                }
            }

            $user->appendLog($request, [
                'old' => $studentOriginal->toArray(),
                'new' => $studentUpdate,
            ]);
            $studentOriginal->update($studentUpdate);


            if ($classRoomId != null) {
                $studentOriginal->classRooms()
                    ->attach($classRoomId);
            }

            if ($deleteClassRoomIds != null && count($deleteClassRoomIds) > 0) {
                foreach ($deleteClassRoomIds as $deleteClassRoomId) {
                    $studentOriginal->classRooms()
                        ->detach($deleteClassRoomId);
                }
            }

            DB::commit();

            $studentOriginal->load([
                'studentParent:id',
                'studentParent.profile:id,user_id,first_name,last_name,address,phone',
                'classRooms:id,academic_level_id,major_id,title,code',
                'image',
            ]);

            return ResponseTools::getInstance()
                ->setMessageFormat('messages.information_was_successfully_updated', ['title' => 'دانش آموز'])
                ->setData('student', $studentOriginal)
                ->getJsonResponse();
        } catch (Throwable $e) {
            DB::rollBack();

            if ($fileUpdate) {
                $fileUpdate->removeFile();
            }

            CommonTools::registerException($e, 'studentUpdate');
        }

        return ResponseTools::getInstance()
            ->setStatus(ResponseTools::RES_ERROR)
            ->setMessageFormat('messages.error_updating_information', ['title' => 'دانش آموز'])
            ->getJsonResponse(Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    public function privateStorage(PrivateStorageStudentRequest $request)
    {
        if ($request->fails()) {
            return ResponseTools::getInstance()
                ->setStatus(ResponseTools::RES_ERROR)
                ->setMessage($request->firstError())
                ->getJsonResponse(Response::HTTP_BAD_REQUEST);
        }

        try {
            $body = CommonTools::safeRequest($request, 'dir', 'filename');

            $localStorage = Storage::disk('local');

            $path = 'students/' . $body['dir'] . '/' . $body['filename'];

            if (!$localStorage->exists($path)) {
                abort(404);
            }

            $stream = $localStorage->readStream($path);
            $mimeType = $localStorage->mimeType($path);
            $fileSize = $localStorage->size($path);

            return response()->stream(
                function () use ($stream) {
                    fpassthru($stream);
                    if (is_resource($stream)) {
                        fclose($stream);
                    }
                },
                200,
                [
                    'Content-Type' => $mimeType,
                    'Content-Length' => $fileSize,
                    'Cache-Control' => 'private, max-age=21600', // 6 ساعت کش مرورگر
                    'Expires' => now()->addHours(6)->toRfc7231String(),
                ]
            );
        } catch (\Throwable $th) {
        }

        abort(404);
    }
}
