<?php

namespace App\Http\Requests\User;

use App\Enums\PermissionUserAccess;
use App\Http\Requests\Model\ModelBaseRequest;
use App\Models\User;
use App\Rules\NationalCodeRule;
use App\Tools\CommonTools;
use App\Tools\Regex;

class UpdateUserRequest extends ModelBaseRequest
{
    protected $authorizePermissions = [
        PermissionUserAccess::update,
        PermissionUserAccess::selfUpdate
    ];


    public function rules()
    {
        $userId = -1;
        $profileId = -1;

        try {
            $userId = CommonTools::toSafeHTML($this->get('id'));
            $profileId = User::findOrFail(intval($userId))
                ->profile
                ->id;
            //code...
        } catch (\Throwable $th) {
            //throw $th;
        }

        return [
            'id' => [
                'required',
                'int',
                'gt:1',
                'exists:users,id'
            ],
            'user.role_id' => [
                'required',
                'int',
                'exists:roles,id',
                'gt:1'
            ],
            'user.active' => ['in:0,1,2'],
            'user.name' => [
                'required',
                'string',
                'min:3',
                'max:100',
                'regex:' . Regex::username,
                "unique_encrypted:users,name,$userId"
            ],
            'user.password' => [
                'confirmed',
                'min:6'
            ],

            'user.profile' => ['required'],
            'user.profile.first_name' => [
                'required',
                'string',
                'min:3',
                'max:100',
                'regex:' . Regex::alphabeticPersian
            ],
            'user.profile.last_name' => [
                'required',
                'string',
                'min:3',
                'max:100',
                'regex:' . Regex::alphabeticPersian
            ],
            'user.profile.national_code' => [
                'required',
                'string',
                'max:10',
                new NationalCodeRule(),
                "unique_encrypted:profiles,national_code,$profileId"
            ],
            'user.profile.address' => [
                'required',
                'string',
                'min:3',
                'max:255',
                'regex:' . Regex::alphabetic
            ],
            'user.profile.phone' => [
                'required',
                'string',
                'min:3',
                'max:12',
                'regex:' . Regex::englishNumber,
                "unique_encrypted:profiles,phone,$profileId"
            ],
            'user.profile.image' => [
                'mimes:jpeg,jpg,png,webp,avif',
                'max:1024'
            ]
        ];
    }
}
