<?php

namespace App\Models;

use App\Http\Requests\Model\ModelBaseRequest;
use App\Tools\CommonTools;
use App\Tools\HasCustomRelations;
use ESolution\DBEncryption\Traits\EncryptedAttribute;
use Exception;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Throwable;

class User extends AuthModel
{
    use HasFactory, EncryptedAttribute, HasCustomRelations;

    public const ACCESS_ADMINISTRATOR = 10;
    public const ACCESS_ADMINISTRATOR_LIMITED = 9;
    public const ACCESS_ADMIN = 20;
    public const ACCESS_ADMIN_LIMITED = 19;
    public const ACCESS_TEACHER = 30;
    public const ACCESS_TEACHER_LIMITED = 29;
    public const ACCESS_PERSONAL_ACCOUNT = 40;
    public const ACCESS_PERSONAL_ACCOUNT_LIMITED = 39;
    public const ACCESS_PARENTS = 50;
    public const ACCESS_PARENTS_LIMITED = 49;
    public const ACCESS_OTHER = 60;
    public const ACCESS_OTHER_LIMITED = 59;
    public const ACCESS_SCHOOL_PRINCIPAL = 70;
    public const ACCESS_SCHOOL_PRINCIPAL_LIMITED = 69;
    public const ACCESS_LIMITED = -1;
    public const ACCESS_DENIED = -2;

    public const ACCESS_LIMITEDS = [
        self::ACCESS_ADMIN_LIMITED,
        self::ACCESS_ADMINISTRATOR_LIMITED,
        self::ACCESS_LIMITED,
        self::ACCESS_TEACHER_LIMITED,
        self::ACCESS_PARENTS_LIMITED,
        self::ACCESS_PERSONAL_ACCOUNT_LIMITED,
    ];

    protected $encryptable = [
        'name',
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'password',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];


    public function profile()
    {
        return $this->hasOne(Profile::class);
    }

    public function teacher()
    {
        return $this->hasOne(Teacher::class);
    }

    public function primaryRole()
    {
        return $this->hasOneThrough(
            Role::class,
            UserRole::class,
            'user_id',
            'roles.id',
            'id',
            'role_id'
        );
    }

    public function getActiveAttribute()
    {
        return $this->userRoles()->first()->primary ?? 0;
    }

    public function userRoles()
    {
        return $this->hasMany(UserRole::class);
    }

    public function roles()
    {
        return $this->belongsToMany(Role::class, 'user_role')
            ->using(UserRole::class)->withPivot('primary')->withTimestamps();
    }

    public function hasPermissions($pPermissionsForCheck)
    {
        $permissionsForCheck = is_array($pPermissionsForCheck) ?
            $pPermissionsForCheck : [$pPermissionsForCheck];

        $permissionsForCheck = array_map(
            [CommonTools::class, 'valueOfEnum'],
            $permissionsForCheck
        );

        $roles = $this->roles()->get();
        $permissions = [];
        foreach ($roles as $role) {
            $permissions = array_merge($permissions, array_column($role->permissions()->get()->toArray(), 'name'));
        }
        return ! empty(array_intersect($permissionsForCheck, $permissions));
    }

    public function getAccessSql($builder, $justAccess = null, $foreignKey = "user_id")
    {
        try {
            $access = $this->getAccess();

            if ($access == self::ACCESS_ADMINISTRATOR && $justAccess != self::ACCESS_PERSONAL_ACCOUNT) {
                return $builder;
            }

            if ($justAccess == self::ACCESS_ADMINISTRATOR) {
                return null;
            }

            if ($access == self::ACCESS_ADMIN && $justAccess != self::ACCESS_PERSONAL_ACCOUNT) {
                return $builder->where($foreignKey, $this->id);
            }

            if ($justAccess == self::ACCESS_ADMIN) {
                return null;
            }

            if ($access == self::ACCESS_TEACHER) {
                return $builder->where($foreignKey, $this->id);
            }

            if ($access == self::ACCESS_PERSONAL_ACCOUNT) {
                return $builder->where($foreignKey, $this->id);
            }
        } catch (Throwable $e) {
        }

        return null;
    }

    public function getAccess(): int
    {
        try {
            $role = $this->roles()
                ->whereIn('primary', [1, 2])
                ->first();

            if ($role->name == 'administrator') {
                $res = self::ACCESS_ADMINISTRATOR;
            } else if (strpos($role->name, 'teacher') !== false) {
                $res = self::ACCESS_TEACHER;
            } else if (strpos($role->name, 'parents') !== false) {
                $res = self::ACCESS_PARENTS;
            } else if (strpos($role->name, 'personalAccount') !== false) {
                $res = self::ACCESS_PERSONAL_ACCOUNT;
            } else if (strpos($role->name, 'schoolPrincipal') !== false) {
                $res = self::ACCESS_SCHOOL_PRINCIPAL;
            } else if (strpos($role->name, 'admin') !== false) {
                $res = self::ACCESS_ADMIN;
            } else {
                $res = self::ACCESS_OTHER;
            }

            if ($role->pivot->primary == 2) {
                $res = $res - 1;
            }
        } catch (Exception $exception) {
            $res = self::ACCESS_DENIED;
            CommonTools::registerException($exception, 'permissionDenied');
        }

        return $res;
    }


    public function logs()
    {
        return $this->hasMany(Log::class);
    }

    public function appendLog(ModelBaseRequest $request, $extra = null)
    {
        $data = [
            'name' => join(',', $request->getPermissionUsedString()),
            'extra' =>
            array_merge(
                [
                    'url' => $request->getPathInfo(),
                ],
                $extra ?? []
            )
        ];
        return $this->logs()
            ->create(CommonTools::sequentialArrayJson($data));
    }

    public function children()
    {
        return $this->hasMany(Student::class, 'parent_id');
    }

    public function schoolSessions()
    {
        return $this->hasMany(SchoolSession::class, 'teacher_user_id');
    }

    public function weeklySchedules()
    {
        return $this->hasMany(WeeklySchedule::class, 'teacher_id');
    }


    public function classRooms()
    {
        return $this->hasManyThrough(
            ClassRoom::class,
            WeeklySchedule::class,
            'teacher_id',
            'id',
            'id',
            'class_room_id'
        );
    }

    public function bells()
    {
        return $this->hasManyThrough(
            Bell::class,
            WeeklySchedule::class,
            'teacher_id',
            'id',
            'id',
            'bell_id',
        );
    }
    public function presences()
    {
        return $this->hasManyThrough(
            Presence::class,
            SchoolSession::class,
            'teacher_user_id',
            'school_session_id',
            'id',
            'id',
        );
    }

    public function getPhoneAttribute()
    {
        return $this->profile->phone;
    }
}
