<?php

namespace App\Rules;

use App\Models\Lesson;
use App\Models\Student;
use App\Models\WeeklySchedule;
use Arr;
use Carbon\Carbon;
use Closure;
use Illuminate\Contracts\Validation\DataAwareRule;
use Illuminate\Contracts\Validation\ValidationRule;
use Morilog\Jalali\Jalalian;

class OwnStudentWithLessonRule implements DataAwareRule, ValidationRule
{
    private $data;
    private $lessonField;
    private $studentField;

    /**
     * validation own students with lesson_id
     * @param string $studentField
     * @param string $lessonField
     */
    public function __construct($studentField, $lessonField)
    {
        $this->studentField = $studentField;
        $this->lessonField = $lessonField;
    }

    /**
     * Run the validation rule.
     *
     * @param  \Closure(string, ?string=): \Illuminate\Translation\PotentiallyTranslatedString  $fail
     */
    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        $tmpLessonField = $this->lessonField;
        if ($groupIdx = strpos($tmpLessonField, '*')) {
            $tmpLessonField = str_replace(
                substr($tmpLessonField, 0, $groupIdx + 2),
                str_replace($this->studentField, '', $attribute),
                $tmpLessonField
            );
        }

        $lessonId =  Arr::get($this->data, $tmpLessonField);

        $realStudentIds = Student::whereHas(
            'classRooms',
            fn($qClassRooms) =>
            $qClassRooms
                ->whereHas(
                    'lessons',
                    fn($qLessons) =>
                    $qLessons
                        ->where(
                            'lessons.id',
                            '=',
                            $lessonId
                        )
                )
        )
            ->where('students.id',  $value)
            ->pluck('id');

        $notAccessStudentIds = array_values(array_diff([$value], $realStudentIds->toArray()));

        if (count($notAccessStudentIds) == 0) {
            return;
        }

        $notAccessStudents = Student::select(['id', 'first_name', 'last_name'])
            ->whereIn('id', $notAccessStudentIds)
            ->get();

        $lesson = Lesson::find($lessonId);

        $fail('validation.own_student_with_lesson', null)
            ->translate([
                'attribute' => __(
                    'validation.attributes',
                )[preg_replace('/\.\d+$/', '.*', $attribute)] ?? $attribute,
                'students' => $notAccessStudents
                    ->map(
                        fn($v) =>
                        $v['first_name'] . ' ' . $v['last_name']
                    )
                    ->join('، '),
                'lesson' => $lesson?->title ?? '-'
            ]);
    }


    public function setData(array $data)
    {
        $this->data = $data;
    }
}
