<?php

namespace App\Rules;

use App\Enums\ApplicationConfigActions;
use App\Models\ApplicationConfig;
use App\Models\Lesson;
use App\Models\Student;
use App\Models\TeachingReport;
use App\Models\WeeklySchedule;
use App\Tools\CommonTools;
use App\Tools\TermTools;
use Arr;
use Carbon\Carbon;
use Closure;
use Exception;
use Illuminate\Contracts\Validation\DataAwareRule;
use Illuminate\Contracts\Validation\ValidationRule;
use Log;
use Morilog\Jalali\Jalalian;

class UniqueTeachingReportRule implements DataAwareRule, ValidationRule
{
    private $data;
    private $lessonField;
    private $studentField;
    private $term;

    /**
     * validation own students with lesson_id
     * @param string $studentField
     * @param string $lessonField
     */
    public function __construct($studentField, $lessonField)
    {
        $this->studentField = $studentField;
        $this->lessonField = $lessonField;

        $ApplicationConfig = ApplicationConfig::whereEncrypted(
            'title',
            '=',
            CommonTools::valueOfEnum(
                ApplicationConfigActions::schoolControllerPrivateTermUpdate
            )
        )
            ->firstOrFail();


        $term = TermTools::getCurrentTerm($ApplicationConfig->value);

        if ($term == null) {
            throw new Exception('not found term');
        }

        $this->term = intval($term['term']);
    }

    /**
     * Run the validation rule.
     *
     * @param  \Closure(string, ?string=): \Illuminate\Translation\PotentiallyTranslatedString  $fail
     */
    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        $tmpLessonField = $attribute . '.' . $this->lessonField;
        $tmpStudentField = $attribute . '.' . $this->studentField;

        $lessonId =  Arr::get($this->data, $tmpLessonField);
        $studentId =  Arr::get($this->data, $tmpStudentField);


        $result = TeachingReport::where(
            'student_id',
            $studentId
        )
            ->where(
                'lesson_id',
                $lessonId
            )
            ->where(
                'term',
                $this->term
            )
            ->where(
                'created_at',
                '>=',
                (new Carbon())->startOfYear()
            )
            ->exists();

        if (!$result) {
            return;
        }

        $student = Student::find($studentId);

        $studentFullName =  $student != null ?
            $student->first_name . ' ' . $student->last_name :
            '-';

        $fail('validation.unique_teaching_report', null)
            ->translate([
                'attribute' => __(
                    'validation.attributes',
                )[preg_replace('/\.\d+$/', '.*', $attribute)] ?? $attribute,
                'student' => $studentFullName
            ]);
    }


    public function setData(array $data)
    {
        $this->data = $data;
    }
}
