<?php

namespace App\Rules;

use Closure;
use Illuminate\Contracts\Validation\ValidationRule;
use Illuminate\Contracts\Validation\DataAwareRule;
use Validator;

class WhenRule implements DataAwareRule, ValidationRule
{

    protected $callbackCondition;
    protected $conditions;
    protected $data;

    /**
     * Creates a new instance of the WhenRule with specified validation conditions.
     *
     * @param callable $callbackCondition function($attribute, $value): string
     * @param array $conditions An associative array of validation rules for files.
     *                          Example: ['then' => ['required', 'max:1024'], 'otherwise' => ['required', 'max:102400']]
     * @return void
     * 
     * ```php
     * new WhenRule(
     * function($attribute, $value){
     * return $value === '1' ? 'then' : 'otherwise'
     * },[
     *     'then' => ['required', 'max:1024'],
     *     'otherwise' => ['required', 'max:102400']
     * ]);
     * ```
     */
    public function __construct($callbackCondition, array $conditions)
    {
        $this->conditions = $conditions;
        $this->callbackCondition = $callbackCondition;
    }

    /**
     * Run the validation rule.
     *
     * @param  string  $attribute
     * @param  mixed  $value
     * @param  Closure  $fail
     * @return void
     */
    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        $action = call_user_func($this->callbackCondition, $attribute, $value) ?? 'otherwise';

        $validator = Validator::make(
            $this->data,
            [
                $attribute => $this->conditions[$action],
            ],
            [],
            [
                'data' => __(
                    'validation.attributes',
                )[preg_replace('/\.\d+$/', '.*', $attribute)] ?? $attribute
            ]
        );

        if ($validator->fails()) {
            $fail($validator->messages()->first());
            return;
        }
    }

    function setData($data)
    {
        $this->data = $data;
    }
}
