<?php

namespace App\Tools;

use Arr;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Log;
use Throwable;

class CommonTools
{
    public static function registerException(
        string|Throwable $message = 'خطایی رخ داده است',
        string $extraInfo = ''
    ): void {
        $message = $message instanceof Throwable &&
            env('APP_LOG_STORE_MODE', 0) == 0 ?
            $message->getMessage() :
            (string) $message;


        Log::error(
            $extraInfo,
            [
                "metadata" => print_r(
                    $message,
                    true
                )
            ]
        );
    }

    public static function toSafeHTML($variable)
    {
        if ($variable == null) {
            return $variable;
        }

        if (CommonTools::isNumericOrBoolOrNull($variable)) {
            return $variable;
        }

        if (is_array($variable)) {
            return array_map(
                function ($item) {
                    if (is_array($item)) {
                        return CommonTools::toSafeHTML($item);
                    }
                    if (CommonTools::isNumericOrBoolOrNull($item)) {
                        return $item;
                    }

                    if (is_string($item)) {
                        return CommonTools::cleanHalfSpace(
                            htmlspecialchars($item)
                        );
                    }

                    return $item;
                },
                $variable
            );
        }

        if (is_string($variable)) {
            return CommonTools::cleanHalfSpace(
                htmlspecialchars($variable)
            );
        }

        return $variable;
    }

    public static function toSafeArray($array, ...$keys)
    {
        return array_intersect_key($array, array_fill_keys($keys, null));
    }

    public static function anyKeyExists($array, ...$keys)
    {

        if ($array === null || count($keys) === 0) {
            return false;
        }

        $keys_to_check = array_flip($keys);

        return  ! empty(array_intersect_key($keys_to_check, $array));
    }

    public static function snakeCase($variable)
    {
        if ($variable == null) {
            return $variable;
        }

        if (CommonTools::isNumericOrBoolOrNull($variable)) {
            return $variable;
        }

        if (is_array($variable)) {
            return array_map(
                function ($item) {
                    if (is_array($item)) {
                        $item = self::snakeCase($item);
                    }
                    return $item;
                },
                self::doSnakeCase($variable)
            );
        }

        if (is_string($variable)) {
            return self::doSnakeCase($variable);
        }
        return $variable;
    }

    public static function isSequentialArray(array $arr): bool
    {
        if (function_exists('array_is_list')) {
            return array_is_list($arr);
        }

        if ([] === $arr) {
            return true;
        }
        return array_keys($arr) === range(0, count($arr) - 1);
    }

    private static function doSnakeCase($variable)
    {
        if (is_array($variable)) {
            $result = [];
            foreach ($variable as $key => $value) {
                $key = strtolower(preg_replace('~(?<=\\w)([A-Z])~', '_$1', $key));
                $result[$key] = $value;
            }
            return $result;
        }

        return strtolower(preg_replace('~(?<=\\w)([A-Z])~', '_$1', $variable));
    }

    public static function isJson($data)
    {
        try {
            if ($data === null) {
                return false;
            }

            $tmp = json_decode($data, true);

            if ($tmp === null) {
                return false;
            }

            return true;
        } catch (\Throwable $ex) {
            return false;
        }
    }

    public static function cleanHalfSpace($text)
    {
        return preg_replace(
            Regex::halfSpace,
            "\xe2\x80\x8c",
            $text
        );
    }

    public static function isNumericOrBoolOrNull($value)
    {
        return is_int($value) || is_float($value) || is_bool($value) || is_null($value);
    }

    public static function enumToString($enum, $exclude = [], $join = ',')
    {
        $values = array_map(fn($case) => $case->value, $enum::cases());

        $filteredValues = array_filter(
            $values,
            fn($value) =>
            !in_array($value, array_map(fn($case) => $case->value, $exclude))
        );

        return implode($join, $filteredValues);
    }

    public static function enumToArray($enum)
    {
        return array_map(fn($case) => $case->value, $enum::cases());
    }

    /**
     * Returns the value of an enum if provided, the string itself if it's a string, or throws an exception otherwise.
     *
     * @param mixed $value
     * @return string
     * @throws \InvalidArgumentException
     */
    public static function valueOfEnum($value): string
    {
        if (is_string($value)) {
            return $value;
        }

        if (is_object($value) && enum_exists(get_class($value))) {
            return $value->value;
        }

        throw new \InvalidArgumentException('The provided value must be either a string or an enum.');
    }

    /**
     * @param FormRequest $request
     * @param mixed ...$keys id, user
     */
    public static function safeRequest(FormRequest $request, mixed ...$keys)
    {

        $value = array_merge($request->route()->parameters(), $request->safe()->all());
        if (count($keys) > 1) {
            $value = self::toSafeArray($value, ...$keys);
        } else if (count($keys) === 1) {
            $value = isset($value[$keys[0]]) ? $value[$keys[0]] : null;
        }


        return CommonTools::toSafeHTML(
            CommonTools::snakeCase(
                $value
            )
        );
    }

    /**
     * remove Classes of array for encode json
     * @return mixed
     */
    public static function sequentialArrayJson(mixed $value)
    {
        if ($value == null) {
            return $value;
        }

        $arr = [...$value];

        $dotValue = Arr::dot($arr);
        $keys = array_keys(Arr::where($dotValue, fn($elValue) => is_object($elValue)));

        Arr::forget($arr, $keys);

        return $arr;
    }
}
