<?php

declare(strict_types=1);

namespace App\Tools;

use App\Exceptions\AnyException;
use App\Models\BaseModel;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Collection;
use InvalidArgumentException;
use Str;
use Throwable;
use Validator;

class DataTableTools
{
    public $columns = null;
    /**
     * @var BaseModel|Collection
     */
    public $data = null;
    public $status = ResponseTools::RES_WARNING;
    public $count = 0;
    public $message = '';
    private $useFirst;
    private $useFilter;
    private $useOrder;
    private $indexQuery = -1;
    private $callbackQuery;

    /**
     * Initializes the Query Filter component, defining which filtering mechanisms
     * should be automatically applied based on the request URL query parameters.
     *
     * @param bool $useFirst Indicates whether pagination parameters (e.g., 'first', 'rows')
     * should be used to limit the results. Default is true (uses parameters like first=0, rows=10).
     * @param bool $useFilter Indicates whether complex filtering parameters
     * (e.g., filter[model][property][*]=value) should be applied to the query. Default is true.
     * @param bool $useOrder Indicates whether ordering parameters
     * (e.g., sort[]:[property, asc|desc] OR sortField:property, sortOrder:asc|desc)
     * should be used to order the results. Default is true.
     */
    public function __construct($useFirst = true, $useFilter = true, $useOrder = true)
    {
        $this->count = 0;
        $this->status = ResponseTools::RES_ERROR;
        $this->message = __('messages.information_was_successfully_no_content');

        $this->useFirst = $useFirst;
        $this->useFilter = $useFilter;
        $this->useOrder = $useOrder;
    }

    /**
     * Set a custom callback function to modify the base query builder instance.
     *
     * This callback allows for modifications such as adding SELECT clauses, JOINs, or other query adjustments
     * and is executed before the main 'where' and 'orderBy' clauses are typically applied.
     *
     * The callback function receives two arguments:
     * 1. $query (Illuminate\Database\Eloquent\Builder or Query\Builder): The current query builder instance.
     * 2. $options (array): An array of first, rows, accurate, sort, sortField, sortOrder, filters.
     *
     * @param callable(
     * \Illuminate\Database\Eloquent\Builder $query,
     * array{first:number,rows:number,accurate:mixed,sort:mixed,sortField:string,sortOrder:mixed,filters:string} $options
     * ): void $callback The callback function used to modify the query builder.
     *
     * @return $this
     */
    public function query($callback)
    {
        $this->callbackQuery = $callback;
        return $this;
    }

    /**
     * Executes the query or queries after applying filters from the request URL.
     *
     * This method supports executing one or multiple queries. If multiple queries are provided,
     * the results are stored in the $this->data property indexed by the 'dataMode' variable,
     * which is initialized to 0 and incremented for each query.
     *
     * @param \Illuminate\Database\Eloquent\Builder|\Illuminate\Database\Eloquent\Builder[] $query
     * The Eloquent query builder instance(s) to be executed (0 to 2 queries are supported).
     * @param \Illuminate\Http\Request $request The current HTTP request instance, containing filtering parameters in the URL query.
     *
     * @return $this
     *
     */
    public function doQuery($query, $request)
    {
        try {
            [$query, $count] = $this->wrapFilters(
                $query,
                $request,
                [
                    'rows' => null,
                    'sortField' =>  'id',
                    'sortOrder' =>  'desc',
                    'withCount' => true,
                ]
            );

            $this->count = $count;

            if ($count > 0) {
                $this->status = ResponseTools::RES_SUCCESS;
                $this->message = __('messages.information_was_successfully_get');
            } else {
                $this->status = ResponseTools::RES_WARNING;
                $this->message = __('messages.information_was_successfully_no_content');
            }

            $this->data = $query->get();
        } catch (Throwable $th) {
            $this->data = null;
            $this->count = 0;
            $this->status = ResponseTools::RES_ERROR;

            if ($th instanceof InvalidArgumentException) {
                $this->message = $th->getMessage();
            } else {
                $this->message = __('messages.error_get_information', ['title' => '-']);
                CommonTools::registerException($th, 'dataTableTools');
            }
        }

        return $this;
    }

    /**
     * Applies filtering, ordering, and pagination constraints from the current HTTP request
     * onto the provided query builder instance.
     *
     * This method utilizes parameters defined in the request URL query (e.g., 'filter', 'sort', 'first', 'rows).
     * The method returns the modified query builder instance without executing the database query.
     *
     * @param \Illuminate\Database\Eloquent\Builder|\Illuminate\Database\Eloquent\Builder[] $query The query builder instance to be modified.
     * @param \Illuminate\Http\Request $request The current HTTP request instance containing filtering parameters.
     * @param array{first:number,rows:number,accurate:mixed,sort:mixed,sortField:string,sortOrder:mixed,filters:string,withCount:bool} $options defined value filters (e.g., 'filter', 'sort', 'first', 'rows).
     *
     * @return \Illuminate\Database\Eloquent\Builder|array{\Illuminate\Database\Eloquent\Builder,number}
     *
     * @throws \Exception May throw exceptions during filter application or validation.
     */
    public function wrapFilters($query, $request, $options = [
        'first' => -1,
        'rows' => null,
        'accurate' => false,
        'sort' => null,
        'sortField' => null,
        'sortOrder' => null,
        'filters' => '',
        'withCount' => false
    ])
    {
        if ($request->fails()) {
            throw new AnyException('validator fail');
        }

        $validateQueries = Validator::make($request->all(), [
            'first' => 'int|min:0',
            'rows' => 'int|min:1',
            'data_mode' => 'int|min:0|max:2',
            'accurate' => 'in:true,false,0,1',
            'sort_field' => 'string',
            'sort_order' => 'string',
            'sort' => 'array|min:1|max:10',
            'sort.*.0' => 'string',
            'sort.*.1' => 'string|in:asc,ASC,desc,DESC',
        ]);

        if (!$validateQueries->passes()) {
            throw new AnyException('filter validation fail');
        }

        $userFilter = collect(
            CommonTools::toSafeHTML(
                CommonTools::snakeCase(
                    $request->all()
                )
            )
        );

        $first = $options['first'] ?? -1;
        $rows = (int) $userFilter->get(
            'rows',
            $options['rows'] ?? null
        );
        $accurate = $options['accurate'] ?? false;
        $sort = $options['sort'] ?? null;
        $sortField = $options['sortField'] ?? null;
        $sortOrder = $options['sortOrder'] ?? null;
        $filters = $options['filters'] ?? '';
        $count = 0;

        if ($this->useFirst) {
            $first = (int) $userFilter->get(
                'first',
                $options['first'] ?? -1
            );
        }

        if ($userFilter->has('accurate')) {
            $accurate = $userFilter->get(
                'accurate',
                $options['accurate'] ?? false
            );
            $accurate = $accurate === 'true' || $accurate == 1 || $accurate == true;
        }

        if ($userFilter->has('sort_order')) {
            $sortOrder = strtolower(
                $userFilter->get(
                    'sort_order',
                    $options['sortOrder'] ?? null
                )
            );
        }

        if ($userFilter->has('sort_field')) {
            $sortField = $userFilter->get('sort_field');
        }

        if ($userFilter->has('sort')) {
            $sort = $userFilter->get('sort');
        }

        if (($userFilter->has('filters')) && ($this->useFilter)) {
            $filters = $userFilter->get(
                'filters',
                $options['sortField'] ?? null
            );
        }

        if (gettype($filters) == 'string') {
            $filters = json_decode($filters, true);
        }

        if (is_array($query)) {
            $this->indexQuery = intval(
                $userFilter->get(
                    'data_mode',
                    0
                )
            );

            $query = $query[count($query) > $this->indexQuery ? $this->indexQuery : 0];

            if (isset($this->callbackQuery)) {
                call_user_func($this->callbackQuery, $query, $sortField);
            }
        }

        $mainTableName = $query->getModel()->getTable();

        if ($filters) {
            if (CommonTools::isSequentialArray($filters)) {
                $query->where(
                    function ($andQuery) use ($filters, $mainTableName, $accurate) {
                        foreach ($filters as $idx => $orFilter) {
                            $andQuery->orWhere(
                                function ($orQuery) use ($orFilter, $mainTableName, $accurate) {
                                    foreach ($orFilter as $model => $groupFields) {
                                        if ($model !== $mainTableName) {
                                            $orQuery->whereHas(
                                                Str::camel($model),
                                                $this->whereQuery($groupFields, $accurate)
                                            );
                                        } else {
                                            $this->whereQuery($groupFields, $accurate)($orQuery);
                                        }
                                    }
                                }
                            );
                        }
                    }
                );
            } else {
                foreach ($filters as $model => $groupFields) {
                    if ($model !== $mainTableName) {
                        $query->whereHas(
                            Str::camel($model),
                            $this->whereQuery($groupFields, $accurate)
                        );
                    } else {
                        $this->whereQuery($groupFields, $accurate)($query);
                    }
                }
            }
        }

        if (($options['withCount'] ?? false) === true) {
            $count = $query->count();
        }

        if (isset($first) && isset($rows) && $first > -1 && $rows > 0) {
            $query->skip($first)->take($rows);
        }

        if ($this->useOrder) {
            if (isset($sort)) {
                foreach ($sort as $vSort) {
                    $query->orderBy($vSort[0], strtolower($vSort[1]));
                }
            } else {
                if (isset($sortField) || isset($sortOrder)) {
                    $query->orderBy($sortField ?: 'id', $sortOrder ?: 'desc');
                }
            }
        }

        if (($options['withCount'] ?? false) === true) {
            return [$query, $count];
        }

        return $query;
    }

    /**
     * Appends specified fields to the resulting Model(s) or Collection(s).
     *
     * Fields can be specified directly (e.g., 'property') or nested,
     * following relationship syntax (e.g., 'model.relationship.property').
     * The method automatically handles single query results and multiple query results
     * based on the $isMultiQuery flag and the internal 'dataMode' index.
     *
     * @param string[]|array<string[]> $fields
     * The field(s) to be appended. Can be a single field string, an array of field strings,
     * or a complex array structure if handling multiple queries.
     * @param bool $isMultiQuery If true, indicates that multiple queries were executed,
     * and the $fields structure must correspond to the indexed 'dataMode' (e.g., $fields[0], $fields[1], etc.).
     *
     * @return $this
     */
    public function appendFields($fields, $isMultiQuery = false)
    {
        $fields = $this->getDataMode($fields, $isMultiQuery);
        ModelTools::appendFields($this->data, $fields);

        return $this;
    }

    private function getDataMode($items, $isMultiQuery = false)
    {
        if ($isMultiQuery && $this->indexQuery !== -1) {
            return $items[count($items) > $this->indexQuery ? $this->indexQuery : 0];
        }

        return $items;
    }

    /**
     * Hide specified fields to the resulting Model(s) or Collection(s).
     *
     * Fields can be specified directly (e.g., 'property') or nested,
     * following relationship syntax (e.g., 'model.relationship.property').
     * The method automatically handles single query results and multiple query results
     * based on the $isMultiQuery flag and the internal 'dataMode' index.
     *
     * @param string[]|array<string[]> $fields
     * The field(s) to be appended. Can be a single field string, an array of field strings,
     * or a complex array structure if handling multiple queries.
     * @param bool $isMultiQuery If true, indicates that multiple queries were executed,
     * and the $fields structure must correspond to the indexed 'dataMode' (e.g., $fields[0], $fields[1], etc.).
     *
     * @return $this
     */
    public function makeHidden($fields, $isMultiQuery = false)
    {
        $fields = $this->getDataMode($fields, $isMultiQuery);
        ModelTools::hideFields($this->data, $fields);

        return $this;
    }

    /**
     * Make visible specified fields to the resulting Model(s) or Collection(s).
     *
     * Fields can be specified directly (e.g., 'property') or nested,
     * following relationship syntax (e.g., 'model.relationship.property').
     * The method automatically handles single query results and multiple query results
     * based on the $isMultiQuery flag and the internal 'dataMode' index.
     *
     * @param string[]|array<string[]> $fields
     * The field(s) to be appended. Can be a single field string, an array of field strings,
     * or a complex array structure if handling multiple queries.
     * @param bool $isMultiQuery If true, indicates that multiple queries were executed,
     * and the $fields structure must correspond to the indexed 'dataMode' (e.g., $fields[0], $fields[1], etc.).
     *
     * @return $this
     */
    public function makeVisible($fields, $isMultiQuery = false)
    {
        $fields = $this->getDataMode($fields, $isMultiQuery);
        if ($fields !== null) {
            [$selfFields, $relationFields] = ModelTools::splitInlineToArray($fields);

            $this->data->makeVisible($selfFields);
            $this->makeRelationVisible($this->data, $relationFields);
        }
        return $this;
    }

    private function makeRelationVisible($data, $treeAppend)
    {
        foreach ($treeAppend as $key => $value) {
            if (count($value) == 0) {
                $data->makeVisible($this->getDataMode($key));
                continue;
            }

            $data->each(function ($col) use ($key, $value) {
                $this->makeRelationVisible($col->$key, $value);
            });
        }
    }

    /**
     * Converts the internal data storage (if it exists) from an Eloquent Collection or Model
     * into a standard PHP array structure.
     *
     * If the internal $this->data property contains an Eloquent Collection or Model,
     * the 'toArray()' method is called on it to convert all models and relationships
     * into nested arrays. The modified data is then stored back in $this->data.
     *
     * @return $this
     */
    public function toArray()
    {
        if ($this->data) {
            $this->data = $this->data->toArray();
        }
        return $this;
    }


    private function whereQuery($groupFields, $accurate, $depth = 1)
    {
        if ($depth > 3) {
            throw new Exception('query very deep');
        }

        return function ($query) use ($groupFields, $accurate, $depth) {
            foreach ($groupFields as $field => $values) {
                if (!CommonTools::isSequentialArray($values)) {
                    $query->whereHas(Str::camel($field), function ($qDeep) use ($values, $accurate, $depth) {
                        $this->whereQuery($values, $accurate, $depth + 1)($qDeep);
                    });
                    continue;
                }

                $query->where(function ($qt) use ($values, $field, $accurate) {
                    foreach ($values as $value) {
                        [$valueSql, $operatorSql, $type] = $this->getSafeOperator($value);

                        if ($type === 'date') {
                            $dateCarbon = Carbon::createFromTimeString(
                                CommonTools::toSafeHTML($valueSql),
                                'UTC'
                            );

                            $qt->orWhere($field, $operatorSql, $dateCarbon);
                            continue;
                        }

                        if ($operatorSql === '=' && ! $accurate) {
                            $operatorSql = 'like';
                            $valueSql = $this->escapeLike('%' . $valueSql . '%');
                        }

                        if ($value === 'null') {
                            $qt->orWhereNull($field);
                            continue;
                        }

                        if ($operatorSql === 'between') {
                            $qt->orWhereBetween($field, $valueSql);
                            continue;
                        }

                        if ($operatorSql === 'in') {
                            $qt->orWhereIn($field, $valueSql);
                            continue;
                        } else if ($operatorSql === 'not in') {
                            $qt->orWhereNotIn($field, $valueSql);
                            continue;
                        }

                        $qt->orWhere($field, $operatorSql, $valueSql);
                        if (method_exists($qt, 'orWhereEncrypted')) {
                            $qt->orWhereEncrypted($field, $operatorSql, $valueSql);
                        }
                    }
                });
            }
        };
    }

    private function getSafeOperator($value)
    {
        $operator = '=';
        $type = 'notDate';

        if (! is_array($value)) {
            return [$value, $operator, $type];
        }

        $v = $value['v'] ?? null;
        $v1 = $value['v1'] ?? null;

        switch ($value['o']) {
            case '>':
                $operator = '>';
                break;

            case '>=':
                $operator = '>=';
                break;

            case '<':
                $operator = '<';
                break;

            case '<=':
                $operator = '<=';
                break;

            case '!':
            case '!=':
            case '<>':
                $operator = '!=';
                break;

            case '~':
            case '~=':
            case 'like':
                $operator = 'like';
                $v = $this->escapeLike($v);
                break;

            case '!~':
            case '!~=':
            case '!like':
            case 'not like':
                $operator = 'not like';
                $v = $this->escapeLike($v);
                break;

            case '[]':
            case 'between':
                if (!isset($value['v'], $value['v1'])) {
                    throw new InvalidArgumentException('Invalid between operator: `v` and `v1` requires');
                }
                $operator = 'between';
                break;

            case 'in':
                if (!is_array($v)) {
                    throw new InvalidArgumentException('IN operator requires array');
                }
                $operator = 'in';
                break;

            case '!in':
            case 'not in':
                if (!is_array($v)) {
                    throw new InvalidArgumentException('NOT IN operator requires array');
                }
                $operator = 'not in';
                break;

            default:
                break;
        }

        if ($operator === 'between') {
            return [[$v, $v1], $operator, $type];
        }

        if (isset($value['t']) && $value['t'] === 'date') {
            $type = 'date';
        }

        return [$v, $operator, $type];
    }

    private function splitInlineToArray($fields)
    {
        [$selfFields, $relationFields] = array_reduce(is_array($fields) ? $fields : [$fields], function ($pValue, $cValue) {
            if (str_contains($cValue, '.')) {
                $pValue[1][] = $cValue;
            } else {
                $pValue[0][] = $cValue;
            }

            return $pValue;
        }, [[], []]);

        return [$selfFields, $this->convertInlineToTree($relationFields)];
    }

    private function convertInlineToTree($array)
    {
        $result = [];

        foreach ($array as $key => $value) {
            $parts = explode(".", $value);

            $node = &$result;
            foreach ($parts as $part) {
                if (! isset($node[$part])) {
                    $node[$part] = [];
                }
                $node = &$node[$part];
            }

            if (count($parts) === 1) {
                $node = $value;
            }
        }

        return $result;
    }

    private function escapeLike(?string $value): string
    {
        $len = strlen($value);
        if (!$value ||  $len == 0) {
            return '';
        }

        if (strpos($value, '%') === false) {
            return $value;
        }

        $isFirst = $value[0] == '%';
        $isLast = $value[$len - 1] == '%';

        if (
            ($len == 1 && $isFirst) ||
            ($len == 2 && $isFirst && $isLast)
        ) {
            return '';
        }

        $tmp = $value;

        if ($isFirst) {
            $tmp =  substr($tmp, 1);
        }

        if ($isLast) {
            $tmp = substr($tmp, 0, strlen($tmp) - 1);
        }

        return ($isFirst ? '%' : '') . addcslashes($tmp, '%_') . ($isLast ? '%' : '');
    }
}
